/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <regex.h>

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_match.h"


int UdmMatchComp(UDM_MATCH * Match,const char * pattern,int match_type,int case_sense,char * errstr,size_t errstrsize){
	regex_t * reg=NULL;
	int flag=REG_EXTENDED;
	int err;
	
	errstr[0]='\0';
	
	switch(match_type){
		case UDM_MATCH_REGEX:
			reg=(regex_t*)malloc(sizeof(regex_t));
			bzero(reg,sizeof(*reg));
			if(case_sense==UDM_MATCH_ICASE)
				flag|=REG_ICASE;
			if((err=regcomp(reg,pattern,flag))){
				regerror(err, reg, errstr, errstrsize);
				free(reg);
				return(1);
			}
			
		case UDM_MATCH_WILD:
		case UDM_MATCH_BEGIN:
		case UDM_MATCH_END:
		case UDM_MATCH_SUBSTR:
		case UDM_MATCH_FULL:
			break;
		default:
			snprintf(errstr,errstrsize,"Unknown match type '%d'",match_type);
			return(1);
	}
	Match->pattern=strdup(pattern);
	Match->match_type=match_type;
	Match->case_sense=case_sense;
	Match->reg=(char*)reg;
	return 0;
}


void UdmMatchFree(UDM_MATCH * Match){
	UDM_FREE(Match->pattern);
	if(Match->reg){
		regfree((regex_t*)Match->reg);
		UDM_FREE(Match->reg);
	}
}

#define UDM_NSUBS 10

int UdmMatchExec(UDM_MATCH * Match, const char * string, size_t nparts,UDM_MATCH_PART * Parts){
	size_t i;
	int res=-1;
	regmatch_t subs[UDM_NSUBS];
	const char * se;
	size_t plen,slen;

	switch(Match->match_type){
		case UDM_MATCH_REGEX:
			if(nparts>UDM_NSUBS)nparts=UDM_NSUBS;
			res=regexec((regex_t*)Match->reg,string,nparts,subs,0);
			if(res){
				for(i=0;i<nparts;i++)Parts[i].beg=Parts[i].end=-1;
			}else{
				for(i=0;i<nparts;i++){
					Parts[i].beg=subs[i].rm_so;
					Parts[i].end=subs[i].rm_eo;
				}
			}
			break;

		case UDM_MATCH_WILD:
			for(i=0;i<nparts;i++)Parts[i].beg=Parts[i].end=-1;
			if(Match->case_sense==UDM_MATCH_ICASE){
				res=UdmStrCaseMatch(string,Match->pattern);
			}else{
				res=UdmStrMatch(string,Match->pattern);
			}
			break;
		case UDM_MATCH_BEGIN:
			for(i=0;i<nparts;i++)Parts[i].beg=Parts[i].end=-1;
			if(Match->case_sense==UDM_MATCH_ICASE){
				res=strncasecmp(Match->pattern,string,strlen(Match->pattern));
			}else{
				res=strncmp(Match->pattern,string,strlen(Match->pattern));
			}
			break;
		case UDM_MATCH_FULL:
			for(i=0;i<nparts;i++)Parts[i].beg=Parts[i].end=-1;
			if(Match->case_sense==UDM_MATCH_ICASE){
				res=strcasecmp(Match->pattern,string);
			}else{
				res=strcmp(Match->pattern,string);
			}
			break;
		case UDM_MATCH_END:
			for(i=0;i<nparts;i++)Parts[i].beg=Parts[i].end=-1;
			plen=strlen(Match->pattern);
			slen=strlen(string);
			if(slen<plen){
				res=1;
				break;
			}
			se=string+slen-plen;
			if(Match->case_sense==UDM_MATCH_ICASE){
				res=strcasecmp(Match->pattern,se);
			}else{
				res=strcmp(Match->pattern,se);
			}
			break;

		case UDM_MATCH_SUBSTR:
		default:
			for(i=0;i<nparts;i++)Parts[i].beg=Parts[i].end=-1;
			res=-1;
	}
	return res;
}

